/**
 * @license
 * Copyright 2022 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */

import fs from 'fs';

import {LH_ROOT} from '../shared/root.js';

/**
 * @param {string} text
 * @param {string|RegExp} searchValue
 * @param {string} replaceValue
 */
function doReplacement(text, searchValue, replaceValue) {
  const newValue = text.replace(searchValue, replaceValue);
  if (newValue === text) throw new Error(`could not find: ${searchValue}`);
  return newValue;
}

/**
 * @param {string} text
 * @param {[string|RegExp, string][]} replacements
 */
function doReplacements(text, replacements) {
  for (const replacement of replacements) {
    text = doReplacement(text, replacement[0], replacement[1]);
  }
  return text;
}

/**
 * @param {string} text
 * @param {string} startPattern
 * @param {string} endPattern
 * @param {[string|RegExp, string][]} replacements
 */
function extract(text, startPattern, endPattern, replacements = []) {
  const startIndex = text.indexOf(startPattern);
  if (startIndex === -1) throw new Error(`could not find: ${startPattern}`);
  const endIndex = text.indexOf(endPattern, startIndex);
  if (endIndex === -1) throw new Error(`could not find: ${endPattern}`);

  const subText = text.substring(startIndex, endIndex + endPattern.length);
  return doReplacements(subText, replacements);
}

/**
 * @param {string} uiStringsDeclare
 * @param {string} extraCode
 */
function createStringsModule(uiStringsDeclare, extraCode) {
  return `
// auto-generated by build/build-cdt-strings.js
/* eslint-disable */

import * as i18n from '../lib/i18n/i18n.js';

${uiStringsDeclare}

const str_ = i18n.createIcuMessageFn(import.meta.url, UIStrings);

${extraCode}`.trimStart();
}

// core/lib/deprecations-strings.js
{
  // eslint-disable-next-line max-len
  const uiStringsFile = `${LH_ROOT}/node_modules/chrome-devtools-frontend/front_end/generated/Deprecation.ts`;
  const generatedDeprecationSourceText = [
    '// auto-generated by build/build-cdt-strings.js\n',
    '/* eslint-disable */\n',
    doReplacements(fs.readFileSync(uiStringsFile, 'utf-8'), [
      [': Partial<Record<string, DeprecationDescriptor>>', ''],
      ['export interface DeprecationDescriptor {', ''],
      ['  milestone?: number;', ''],
      ['  chromeStatusFeature?: number;\n}', ''],
      // Some patterns are supported in DevTools UIStrings, but not ours.
      [/\\\\/g, ''],
      [`{imageOrientation: 'from-image'}`, `\\\\{imageOrientation: 'from-image'\\\\}`],
    ]),
  ].join('');
  fs.writeFileSync(`${LH_ROOT}/core/lib/deprecations-strings.js`, generatedDeprecationSourceText);
}

// core/lib/bf-cache-strings.js
{
  // eslint-disable-next-line max-len
  const inFile = `${LH_ROOT}/node_modules/chrome-devtools-frontend/front_end/panels/application/components/BackForwardCacheStrings.ts`;
  const outFile = `${LH_ROOT}/core/lib/bf-cache-strings.js`;

  const input = fs.readFileSync(inFile, 'utf-8');

  const uiStringsDeclare = extract(input, 'const UIStrings', '};');
  const notRestoredReasonDescriptionDeclare =
    extract(input, 'const NotRestoredReasonDescription', '};', [
      [/i18nLazyString/g, 'str_'],
    ]);

  const extraCode = `
/** @type {Record<string, {name: LH.IcuMessage} | undefined>} */
${notRestoredReasonDescriptionDeclare}

export {
  NotRestoredReasonDescription,
  UIStrings,
};`;
  fs.writeFileSync(outFile, createStringsModule(uiStringsDeclare, extraCode));
}
