/**
 * @license
 * Copyright 2022 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */

/** @type {LH.Config} */
const config = {
  extends: 'lighthouse:default',
  categories: {
    performance: {
      title: 'Performance',
      auditRefs: [
        {id: 'oopif-iframe-test-audit', weight: 0},
        {id: 'script-elements-test-audit', weight: 0},
      ],
    },
  },
  audits: [
    // Include an audit that *forces* the IFrameElements artifact to be used for our test.
    {path: 'oopif-iframe-test-audit'},
    {path: 'script-elements-test-audit'},
  ],
  settings: {
    // This test runs in CI and hits the outside network of a live site.
    // Be a little more forgiving on how long it takes all network requests of several nested iframes
    // to complete.
    maxWaitForLoad: 180000,
    // CI machines are pretty weak which lead to many more long tasks than normal.
    // Reduce our requirement for CPU quiet.
    cpuQuietThresholdMs: 500,
  },
};

/**
 * @type {Smokehouse.ExpectedRunnerResult}
 * Expected Lighthouse audit values for sites with OOPIFS.
 */
const expectations = {
  lhr: {
    requestedUrl: 'http://localhost:10200/oopif-scripts.html',
    finalDisplayedUrl: 'http://localhost:10200/oopif-scripts.html',
    audits: {
      'network-requests': {
        details: {
          items: {
            // Multiple session attach handling fixed in M105
            // https://chromiumdash.appspot.com/commit/f42337f1d623ec913397610ccf01b5526e9e919d
            _minChromiumVersion: '105',
            _includes: [
              {url: 'http://localhost:10200/oopif-scripts.html'},
              {url: 'http://localhost:10200/oopif-simple-page.html'},
              {url: 'http://localhost:10503/oopif-simple-page.html'},
              // simple-script.js is included many times
              // 2 * (1 from <script>, 1 from fetch) = 4
              // Note, the network records from the workers are _not_ captured! If they
              // were, then we would see 8 simple-script.js
              {url: 'http://localhost:10200/simple-script.js', resourceType: 'Script'},
              {url: 'http://localhost:10503/simple-script.js', resourceType: 'Script'},
              {url: 'http://localhost:10200/simple-script.js', resourceType: 'Fetch'},
              {url: 'http://localhost:10503/simple-script.js', resourceType: 'Fetch'},
              {url: 'http://localhost:10200/simple-worker.js'},
              {url: 'http://localhost:10503/simple-worker.js'},
              {url: 'http://localhost:10200/simple-worker.mjs'},
              {url: 'http://localhost:10503/simple-worker.mjs'},
              // Requests from worker targets
              {url: 'http://localhost:10200/simple-script.js?esm', resourceType: 'Script'},
              {url: 'http://localhost:10503/simple-script.js?esm', resourceType: 'Script'},
              {url: 'http://localhost:10200/simple-script.js?importScripts', resourceType: 'Other'},
              {url: 'http://localhost:10503/simple-script.js?importScripts', resourceType: 'Other'},
            ],
            // Ensure the above is exhaustive (except for favicon, which won't be fetched in devtools/LR).
            _excludes: [
              {url: /^((?!favicon).)*$/s},
            ],
          },
        },
      },
    },
  },
  artifacts: {
    IFrameElements: [
      {
        id: 'iframe-1',
        src: 'http://localhost:10200/oopif-simple-page.html',
        clientRect: {
          width: '>0',
          height: '>0',
        },
        isPositionFixed: true,
      },
      {
        id: 'iframe-2',
        src: 'http://localhost:10503/oopif-simple-page.html',
        clientRect: {
          width: '>0',
          height: '>0',
        },
        isPositionFixed: true,
      },
    ],
    // Only `:10200/oopif-simple-page.html`'s inclusion of `simple-script.js` shows here.
    // All other scripts are filtered out because of our "OOPIF" filter.
    ScriptElements: [
      {
        src: 'http://localhost:10200/simple-script.js',
        source: 'network',
      },
      {
        // Worker requests emitted on the worker's parent target since M121.
        _minChromiumVersion: '121',
        src: 'http://localhost:10200/simple-worker.mjs',
        source: 'network',
      },
      {
        // Worker requests emitted on the worker's parent target since M121.
        _minChromiumVersion: '121',
        src: 'http://localhost:10200/simple-worker.js',
        source: 'network',
      },
    ],
    // Same here, except we get inline scripts of the iframe.
    Scripts: {
      _includes: [
        {
          url: 'http://localhost:10200/simple-script.js',
          content: /🪁/,
        },
        {
          url: 'http://localhost:10200/oopif-simple-page.html',
          content: /new Worker/,
        },
      ],
      _excludes: [{}],
    },
  },
};

export default {
  id: 'oopif-scripts',
  expectations,
  config,
};
