/**
 * @license
 * Copyright 2017 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */

import assert from 'assert/strict';

import SplashScreenAudit from '../../audits/splash-screen.js';
import {parseManifest} from '../../lib/manifest-parser.js';
import {readJson} from '../test-utils.js';

const manifest = readJson('../fixtures/manifest.json', import.meta);
const manifestDirtyJpg = readJson('../fixtures/manifest-dirty-jpg.json', import.meta);

const manifestSrc = JSON.stringify(manifest);
const manifestDirtyJpgSrc = JSON.stringify(manifestDirtyJpg);
const EXAMPLE_MANIFEST_URL = 'https://example.com/manifest.json';
const EXAMPLE_DOC_URL = 'https://example.com/index.html';

/**
 * @param {string} src
 */
function generateMockArtifacts(src = manifestSrc) {
  const exampleManifest = parseManifest(src, EXAMPLE_MANIFEST_URL, EXAMPLE_DOC_URL);

  return {
    WebAppManifest: exampleManifest,
    InstallabilityErrors: {errors: []},
  };
}
function generateMockAuditContext() {
  return {
    computedCache: new Map(),
  };
}
describe('PWA: splash screen audit', () => {
  describe('basics', () => {
    it('fails if page had no manifest', () => {
      const artifacts = generateMockArtifacts();
      artifacts.WebAppManifest = null;
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('No manifest was fetched'), result.explanation);
      });
    });

    it('fails with a non-parsable manifest', () => {
      const artifacts = generateMockArtifacts('{,:}');
      const context = generateMockAuditContext();
      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('failed to parse as valid JSON'));
      });
    });

    it('fails when an empty manifest is present', () => {
      const artifacts = generateMockArtifacts('{}');
      const context = generateMockAuditContext();
      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation);
        assert.strictEqual(result.details.items[0].failures.length, 4);
      });
    });

    it('passes with complete manifest and SW', () => {
      const context = generateMockAuditContext();
      return SplashScreenAudit.audit(generateMockArtifacts(), context).then(result => {
        assert.strictEqual(result.score, 1, result.explanation);
        assert.strictEqual(result.explanation, undefined, result.explanation);
      });
    });
  });

  describe('one-off-failures', () => {
    it('fails when a manifest contains no name', () => {
      const artifacts = generateMockArtifacts();
      artifacts.WebAppManifest.value.name.value = undefined;
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('name'), result.explanation);
      });
    });

    it('fails when a manifest contains no background color', () => {
      const artifacts = generateMockArtifacts();
      artifacts.WebAppManifest.value.background_color.value = undefined;
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('background_color'), result.explanation);
      });
    });

    it('fails when a manifest contains no theme color', () => {
      const artifacts = generateMockArtifacts();
      artifacts.WebAppManifest.value.theme_color.value = undefined;
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('theme_color'), result.explanation);
      });
    });

    it('fails if page had no icons in the manifest', () => {
      const artifacts = generateMockArtifacts();
      artifacts.WebAppManifest.value.icons.value = [];
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('PNG icon'), result.explanation);
      });
    });

    it('fails if icons were present, but no valid PNG present', () => {
      const artifacts = generateMockArtifacts(manifestDirtyJpgSrc);
      const context = generateMockAuditContext();

      return SplashScreenAudit.audit(artifacts, context).then(result => {
        assert.strictEqual(result.score, 0);
        assert.ok(result.explanation.includes('PNG icon'), result.explanation);
        const failures = result.details.items[0].failures;
        assert.strictEqual(failures.length, 1, failures);
      });
    });
  });
});
